// FieldGuardian Compliance Overlay Content Script

// Main initialization function
function init() {
  console.log('FieldGuardian Compliance Overlay initialized');
  
  // Scan for form fields when the extension is first loaded
  scanForFields();
  
  // Set up observer to detect dynamically loaded content
  observeDOMChanges();
}

// Observe DOM changes to detect new fields
function observeDOMChanges() {
  const observer = new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type === 'childList' && mutation.addedNodes.length > 0) {
        // Delay scan to allow DOM to settle
        setTimeout(scanForFields, 300);
        break;
      }
    }
  });
  
  // Start observing the document body
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

// Scan the page for input fields
function scanForFields() {
  // Look for textarea elements, which are commonly used for responses
  const textareas = document.querySelectorAll('textarea');
  textareas.forEach(textarea => {
    // Skip if already processed
    if (textarea.dataset.fieldguardianProcessed) return;
    
    // Mark as processed
    textarea.dataset.fieldguardianProcessed = 'true';
    
    // Add a check button next to the textarea
    addCheckButtonToField(textarea);
  });
}

// Add check button to field
function addCheckButtonToField(field) {
  // Create check button element
  const checkButton = document.createElement('button');
  checkButton.textContent = 'Check Compliance';
  checkButton.className = 'fieldguardian-check-button';
  checkButton.style.cssText = `
    position: absolute;
    top: 5px;
    right: 5px;
    z-index: 9999;
    background-color: #2563eb;
    color: white;
    border: none;
    border-radius: 4px;
    padding: 5px 10px;
    font-size: 12px;
    cursor: pointer;
  `;
  
  // Add click event listener
  checkButton.addEventListener('click', (e) => {
    e.preventDefault();
    e.stopPropagation();
    
    // Get field value
    const text = field.value;
    
    // Ensure we have text to analyze
    if (!text || text.trim().length < 20) {
      alert('Please enter more text before checking compliance (minimum 20 characters).');
      return;
    }
    
    // Show loading state
    checkButton.textContent = 'Analyzing...';
    checkButton.disabled = true;
    
    // Send message to background script for analysis
    chrome.runtime.sendMessage(
      { action: 'analyze', text },
      (response) => {
        // Reset button state
        checkButton.textContent = 'Check Compliance';
        checkButton.disabled = false;
        
        // Show mock result (in real extension, this would display actual analysis)
        showMockResult(field);
      }
    );
  });
  
  // Add button to a wrapper around the field
  const wrapper = document.createElement('div');
  wrapper.style.cssText = 'position: relative; display: inline-block;';
  
  // Insert wrapper before field
  field.parentNode.insertBefore(wrapper, field);
  
  // Move field into wrapper
  wrapper.appendChild(field);
  
  // Add button to wrapper
  wrapper.appendChild(checkButton);
}

// Show mock result (for demo purposes)
function showMockResult(field) {
  // Create results container
  const resultsContainer = document.createElement('div');
  resultsContainer.className = 'fieldguardian-results';
  resultsContainer.style.cssText = `
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 400px;
    max-width: 90vw;
    background-color: white;
    border-radius: 8px;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.2);
    padding: 20px;
    z-index: 10000;
  `;
  
  // Add header
  const header = document.createElement('h2');
  header.textContent = 'Compliance Analysis';
  header.style.cssText = 'margin-top: 0; color: #1e3a8a; font-size: 18px;';
  resultsContainer.appendChild(header);
  
  // Add rating
  const rating = document.createElement('div');
  rating.innerHTML = '<strong>Rating:</strong> Partially Compliant';
  rating.style.cssText = 'margin-bottom: 10px; color: #ca8a04;';
  resultsContainer.appendChild(rating);
  
  // Add score
  const score = document.createElement('div');
  score.innerHTML = '<strong>Score:</strong> 0.68';
  score.style.cssText = 'margin-bottom: 10px;';
  resultsContainer.appendChild(score);
  
  // Add improvements
  const improvements = document.createElement('div');
  improvements.innerHTML = '<strong>Suggested Improvements:</strong>';
  improvements.style.cssText = 'margin-bottom: 10px;';
  resultsContainer.appendChild(improvements);
  
  // Add improvement list
  const improvementList = document.createElement('ul');
  improvementList.style.cssText = 'margin-top: 5px; padding-left: 20px;';
  improvementList.innerHTML = `
    <li>Include specific reference to SOP-QA-023 for deviation handling</li>
    <li>Add explicit timeline for CAPA implementation</li>
  `;
  improvements.appendChild(improvementList);
  
  // Add close button
  const closeButton = document.createElement('button');
  closeButton.textContent = 'Close';
  closeButton.style.cssText = `
    background-color: #e5e7eb;
    border: none;
    border-radius: 4px;
    padding: 8px 16px;
    margin-right: 10px;
    cursor: pointer;
  `;
  closeButton.addEventListener('click', () => {
    document.body.removeChild(resultsContainer);
  });
  
  // Add apply suggestions button
  const suggestButton = document.createElement('button');
  suggestButton.textContent = 'Apply Suggestions';
  suggestButton.style.cssText = `
    background-color: #2563eb;
    color: white;
    border: none;
    border-radius: 4px;
    padding: 8px 16px;
    cursor: pointer;
  `;
  suggestButton.addEventListener('click', () => {
    // In a real extension, this would modify the text with AI suggestions
    field.value += '\n\nAs per SOP-QA-023, we will implement the CAPA within 30 days of approval.';
    document.body.removeChild(resultsContainer);
  });
  
  // Add buttons container
  const buttonsContainer = document.createElement('div');
  buttonsContainer.style.cssText = 'display: flex; justify-content: flex-end; margin-top: 20px;';
  buttonsContainer.appendChild(closeButton);
  buttonsContainer.appendChild(suggestButton);
  resultsContainer.appendChild(buttonsContainer);
  
  // Add to document
  document.body.appendChild(resultsContainer);
}

// Initialize the extension
init();